/* Copyright (C) 2014-2018 RealVNC Ltd.  All Rights Reserved.
 */

#ifndef UUID_d58c4368_01dd_11e4_b948_0b350bf95e7d
#define UUID_d58c4368_01dd_11e4_b948_0b350bf95e7d

/**
 * \file vncwifidisplayparameters.h
 *
 * This header file contains macros that expand to the names of VNC WiFi
 * Display Sink parameters. These are for use in calls to
 * VNCWiFiDisplaySinkSetParameter() and VNCWiFiDisplaySinkGetParameter().
 *
 * All parameter values are passed to and from the SDK as null-terminated
 * UTF-8 strings. If a parameter is listed as having an integer value, then the
 * string should contain the value formatted in decimal.  If a parameter is
 * listed as having a Boolean value, the string should have the value "0" or
 * "1".
 *
 * Macros are also provided which define the default value of the parameters.
 * For example, the default value of VNCWiFiDisplayParameterLog is given by the
 * VNCWiFiDisplayDefaultValueLog macro.
 *
 * \see VNCWiFiDisplaySinkSetParameter, VNCWiFiDisplaySinkGetParameter
 */

/**
 * \brief Controls the SDK's logging.
 *
 * If the value is an empty string, then all logging is disabled.  If the
 * string is non-empty, then it is a comma-separated list of directives of the
 * form category:severity.
 *
 * category may be "*", which means that the selected severity is to be used
 * for all log categories. The category identifies the internal component
 * reporting the log message. Normally, you'll want to include all log
 * categories.
 *
 * severity is an integer from 0 to 100 inclusive.  The SDK defines these
 * logging severities, but may use other values in between:
 *
 *  -   0 - serious errors
 *  -  10 - warning and connection status messages
 *  -  30 - informational messages
 *  - 100 - verbose debugging
 *
 * Changes to this parameter will take effect even if the WiFi Display sink
 * thread is already running.
 *
 * String.
 *
 * \see VNCWiFiDisplaySinkLogCallback
 * \see VNCWiFiDisplayDefaultValueLog
 */
#define VNCWiFiDisplayParameterLog "Log"

/**
 * \brief The default value for the logging parameter.
 *
 * With this value, logging is disabled.
 *
 * \see VNCWiFiDisplayParameterLog
 */
#define VNCWiFiDisplayDefaultValueLog ""

/**
 * \brief Sets the UDP port that a VNC WiFi Display Sink should listen on for
 * RTP streams.
 *
 * This specifies the UDP port that the VNC WiFi Display Sink's RTP client
 * shall listen on for RTP streams. The UDP port must be available for use by
 * the SDK. If multiple VNC WiFi Display Sinks are being used, then each
 * instance must use a different port.
 *
 * The RTP specification mandates that only even-numbered ports should be
 * used. Therefore, odd-numbered ports will be rejected for this parameter.
 *
 * If this parameter is set to zero, then the SDK shall choose an available UDP
 * port.
 *
 * Integer, minimum 0, maximum 65534.
 */
#define VNCWiFiDisplayParameterSinkRTPListenPort "SinkRTPListenPort"

/**
 * \brief The default value for the sink's RTP UDP listen port.
 *
 * With this value, the SDK chooses an available UDP port.
 */
#define VNCWiFiDisplayDefaultValueSinkRTPListenPort "0"

/**
 * \brief The size (in characters) to limit cut text to before sending to a
 * MirrorLink WiFi Display source.
 *
 * Sink cut text events may be sent over a UIBC connection to a MirrorLink
 * WiFi Display source. It is generally undesirable to send large cut text
 * due to the time taken to transfer the data as well as memory usage
 * considerations within the receiving MirrorLink WiFi Display source. By
 * truncating the cut text to an upper limit of bytes before sending then the
 * adverse performance impact of large blocks of cut text can be avoided.
 *
 * If the value is non-zero, then a single call to
 * VNCWiFiDisplaySinkSendMirrorLinkSinkCutTextEvent() will never send more than
 * this number of UTF-8 characters to the MirrorLink WiFi Display source. The
 * remaining text is silently discarded.
 *
 * If the value is zero, then
 * VNCWiFiDisplaySinkSendMirrorLinkSinkCutTextEvent() will always send all of
 * the text to the MirrorLink WiFi Display source.
 *
 * Integer, minimum 0.
 */
#define VNCWiFiDisplayParameterMaxCutText "MaxCutText"

/**
 * \brief The default value for the cut text size limit.
 */
#define VNCWiFiDisplayDefaultValueMaxCutText "16384"

/**
 * \brief The directory in the local filesystem from which the SDK will attempt
 * to load decoder plugin DLLs and shared objects.
 *
 * Whenever VNCWiFiDisplaySinkLoadDecoderFromFile() is called, the SDK attempts
 * to load the decoder from a DLL or shared object. The filename is
 * 'vncwifidisplaydecoder-NAME.dll' on Windows,
 * 'libvncwifidisplaydecoder-NAME.dylib' on Mac OS X, or 
 * 'libvncwifidisplaydecoder-NAME.so' on UNIX, where NAME is the string
 * provided to the API call. The directory in which the DLL or shared object is
 * expected to exist is given by the value of this parameter.
 *
 * Changes to this parameter after the WiFi Display sink thread has started
 * will have no effect.
 *
 * An empty value indicates that the system library search method should be
 * used.
 *
 * A value of "." will use the current working directory.
 *
 * String.
 *
 * \see VNCWiFiDisplaySinkLoadDecoderFromFile
 */
#define VNCWiFiDisplayParameterDecoderDirectory "DecoderDirectory"

/**
 * \brief The default value for the decoder directory parameter.
 *
 * With this value, the system library search method shall be used.
 */
#define VNCWiFiDisplayDefaultValueDecoderDirectory ""

/**
 * \brief The maximum RTP packet size expected to be received, in bytes.
 *
 * The SDK will allocate packet buffers of this size for the purpose of
 * receiving RTP data. Packets larger than this will be discarded.
 *
 * Changes to this parameter after the WiFi Display sink thread has started
 * will have undefined behaviour.
 *
 * Integer.
 */
#define VNCWiFiDisplayParameterRTPMTU "rtpMtu"

/**
 * \brief The default value for the RTP MTU.
 */
#define VNCWiFiDisplayDefaultValueRTPMTU "1500"

/**
 * \brief The system UDP buffer size to specify for the RTP socket.
 *
 * Note that the behaviour of this parameter is highly platform specific.
 * Choosing a value which is too low can result in dropped RTP packets.
 *
 * Changes to this parameter after the WiFi Display sink thread has started
 * will have undefined behaviour.
 *
 * Integer.
 */
#define VNCWiFiDisplayParameterRTPUDPBufferSize "rtpUdpBufferSize"

/**
 * \brief The default value for the RTP UDP buffer size (1MB).
 *
 * On QNX platforms, this is 120K due to a restricted receive buffer size.
 * On all other supported platforms, this is 1MB.
 */
#ifdef __QNX__
#define VNCWiFiDisplayDefaultValueRTPUDPBufferSize "122880"
#else
#define VNCWiFiDisplayDefaultValueRTPUDPBufferSize "1048576"
#endif

/**
 * \brief The period between retrying RTSP connections.
 *
 * When attempting to establish an RTSP connection the WiFi Display
 * sink will periodically retry the TCP connection for a period of 90
 * seconds from the request attempt. This parameter controls the
 * period between retries.
 *
 * The value is an integer string specifying the number of
 * milliseconds between TCP connection attempts for the RTSP
 * connection.
 *
 * Integer, minimum 1, maximum 89999.
 *
 * \see VNCWiFiDisplayDefaultValueRTSPConnectRetryPeriod
 */
#define VNCWiFiDisplayParameterRTSPConnectRetryPeriod "rtspConnectRetryPeriod"

/**
 * \brief The default value for the RTSP connect retry period.
 */
#define VNCWiFiDisplayDefaultValueRTSPConnectRetryPeriod "1000"


#endif  // UUID_d58c4368_01dd_11e4_b948_0b350bf95e7d
